//******************************************************************************
//   MSP-5xx Demo - SMBus Master transmits & receives from SMBus Slave
//
//   Description: This demo connects two MSP430's via the I2C/SMBus. The master
//   transmits 1 byte to the slave, then recieves 1 byte from the slave.  This
//   is the master code.  The slave code for successful transactions is called 
//   SMBus_Slv_2.c.  This can double as an "SMBus read byte" command to a TI 
//   gas gauge part (BQxxxx). LED on P1.0 blinks when a timeout occurs
//  
//   The slave code that demonstrates a timeout condition is 
//   msp430x5xx_SMB_SLV_timeout.c
//   
//   ACLK = 32kHz, MCLK = SMCLK = default DCO = ~1.045MHz
//   
//                                 /|\  /|\
//                 MSP430F5438     10k  10k    MSP430F5438
//                    slave         |    |        master           
//             |-----------------|  |    | |-----------------| 
//             |             P3.7|<-|----+>|P3.7         P1.0|-->LED
//             |                 |  |      |                 |
//             |             P5.4|<-+----->|P5.4             |
//             |                 |         |                 |
//             |             P2.0|<------->|P2.0             |
//
//
//  J. Kam/H. Grewal
//  Texas Instruments Inc.
//  July 2009
//  Built with Code Composer Essentials v3.1 Build: 3.2.4.3.8 
//******************************************************************************

#include "msp430x54x.h"

#define COUNTS_25MS     763                 // 0.025s/32768s = 763

unsigned char data_to_send = 5;
unsigned char data_received[1];
unsigned char *PTxData;                     // Pointer to TX data
unsigned char TXByteCtr;
unsigned char *PRxData;                     // Pointer to RX data
unsigned char RXByteCtr;
unsigned char errorFlag=0;
unsigned char doneFlag=0;

void i2c_init(void);

void main(void)
{
  WDTCTL = WDTPW + WDTHOLD;                 // Stop WDT

  i2c_init();                               // Initializes USCI for I2C
  UCSCTL4 |= SELA__REFOCLK;
  
  P1DIR |= BIT0;                            // LED output
  P1OUT &= ~BIT0;
  
  P2OUT &= ~BIT0;                           // Active high
  P2DIR |= BIT0;                            // Output for resetting slave  
    
  __enable_interrupt();
  
  TA1CCR0 = COUNTS_25MS;
  TA1CCTL0 = CCIE;                          // Set timeout interrupt  
  
  while(1)
  {
    TXByteCtr = 1;
    PTxData = &(data_to_send);

    RXByteCtr = 1;
    PRxData = &(data_received[0]);
    while(UCB1CTL1 & UCTXSTP);              // Stop condition sent?
    TA1CTL = TASSEL__ACLK + TACLR + MC__UP;
    UCB1CTL1 |= UCTR + UCTXSTT;             // Go!
    __bis_SR_register(CPUOFF);              // Enter LPM0 w/ interrupts
  }
}

void i2c_init(void)
{
  P5SEL |= BIT4 + BIT5;                     // Assigns I2C pins on 5438
  P3SEL |= BIT7;
  P3SEL &= ~BIT6;
  P3DIR |= BIT6;                            // Create a slave select line
  UCB1CTL1 |= UCSWRST;                      // Enable SW reset
  UCB1CTL0 = UCMST + UCMODE_3 + UCSYNC;     // I2C Master, synchronous mode
  UCB1CTL1 = UCSSEL_2 + UCSWRST;            // Use SMCLK, keep SW reset
  UCB1BR0 = 10;                             // fSCL = SMCLK/10 ~ 100kHz
  UCB1BR1 = 0;
  UCB1I2CSA = 0x48;                         // Slave Address is 1st byte of PayLoad
  UCB1CTL1 &= ~UCSWRST;                     // Clear SW reset, resume operation
  UCB1IE |= UCTXIE + UCRXIE + UCNACKIE;     // Enable TX, RX, NACK interrupt
}

/*----------------------------------------------------------------------------+
| I2C/SMBus interrupt (USCI B1)
+----------------------------------------------------------------------------*/
#pragma vector = USCI_B1_VECTOR
__interrupt void USCI_B1_ISR(void)
{
  switch(__even_in_range(UCB1IV, 12))
  {
    case 0:
      break;
    case 2:                                 // ALIFG
      break;
    case 4:                                 // NACKIFG
      UCB1CTL1 |= UCTXSTP;                  // Transmit a stop
      UCB1IFG &= ~UCNACKIFG;                // Clear NACK flag
      TA1CTL = MC__STOP;                    // Turn off timeout timer
      errorFlag = 2;
      break;
    case 6:                                 // STTIFG
      break;
    case 8:                                 // STPIFG
      break;
    case 10:                                // RXIFG
      TA1CTL = MC__STOP;                    // Turn off timeout timer
      RXByteCtr--;                          // Decrement RX byte counter
      
      if (RXByteCtr)
      {
        *PRxData++= UCB1RXBUF;              // Multiple bytes left to receive
        TA1CTL = TASSEL__ACLK + TACLR + MC__UP;
      }
      else
      {
        *PRxData= UCB1RXBUF;                // Last byte to be received
        __bic_SR_register_on_exit(CPUOFF);  // Exit LPM0
        doneFlag=1;                         // Flag to show read completed
      }

      if (RXByteCtr == 1)                   // Only one byte left? 
        UCB1CTL1 |= UCTXSTP;                // Generate I2C stop condition
                                            // If reading only 1 byte, stop condition
                                            // should have already been sent
      break; 
    case 12:                                // TXIFG
                                            // Set when UCB1TXBUF is empty

      TA1CTL = MC__STOP;                    // Turn off timeout timer
      if (TXByteCtr)                        // If there's something to transmit
      {
        UCB1TXBUF = *PTxData++;             // Load TX buffer
        TXByteCtr--;                        // Decrement TX byte counter
      }
      else                                  // Nothing more to transmit 
      {                                      
        UCB1CTL1 &= ~UCTR;                  // Receiver mode
        UCB1IFG &= ~UCTXIFG;                // Clear USCI_B0 TX int flag
        UCB1CTL1 |= UCTXSTT;                // I2C restart (SMBus protocol)
        
        while(UCB1CTL1 & UCTXSTT);          // Start condition sent?
          UCB1CTL1 |= UCTXSTP;              // Send stop signal immediately (during first byte)
      }
      TA1CTL = TASSEL__ACLK + TACLR + MC__UP;
      break;
    default:
      break;
  }
}

#pragma vector=TIMER1_A0_VECTOR
__interrupt void TIMER1_A0_ISR(void)
{
  TA1CTL = MC__STOP;                        // Turn off timeout timer
  UCB1CTL1 |= UCTXSTP;                      // Send stop on SDA line
  P2OUT |= BIT0;
  P1OUT ^= BIT0;                            // Toggle LED if timeout occured
  UCB1IFG &= ~UCTXIFG;                      // Clear TX & RX interrupt flags
  UCB1IFG &= ~UCRXIFG;                      // so no more bytes are sent/received
  UCB1CTL1 |= UCSWRST;                      // Enable SW reset
  UCB1CTL1 &= ~UCSWRST;
  errorFlag = 1;                            // Set timeout flag
}

